/*
 * Decompiled with CFR 0.152.
 */
package com.datastax.oss.driver.internal.core.session.throttling;

import com.datastax.oss.driver.api.core.RequestThrottlingException;
import com.datastax.oss.driver.api.core.config.DefaultDriverOption;
import com.datastax.oss.driver.api.core.config.DriverExecutionProfile;
import com.datastax.oss.driver.api.core.context.DriverContext;
import com.datastax.oss.driver.api.core.session.throttling.RequestThrottler;
import com.datastax.oss.driver.api.core.session.throttling.Throttled;
import com.datastax.oss.driver.internal.core.context.InternalDriverContext;
import com.datastax.oss.driver.internal.core.session.throttling.NanoClock;
import com.datastax.oss.driver.shaded.guava.common.annotations.VisibleForTesting;
import edu.umd.cs.findbugs.annotations.NonNull;
import io.netty.util.concurrent.EventExecutor;
import java.time.Duration;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class RateLimitingRequestThrottler
implements RequestThrottler {
    private static final Logger LOG = LoggerFactory.getLogger(RateLimitingRequestThrottler.class);
    private final String logPrefix;
    private final NanoClock clock;
    private final int maxRequestsPerSecond;
    private final int maxQueueSize;
    private final long drainIntervalNanos;
    private final EventExecutor scheduler;
    private final ReentrantLock lock = new ReentrantLock();
    @GuardedBy(value="lock")
    private long lastUpdateNanos;
    @GuardedBy(value="lock")
    private int storedPermits;
    @GuardedBy(value="lock")
    private final Deque<Throttled> queue = new ArrayDeque<Throttled>();
    @GuardedBy(value="lock")
    private boolean closed;

    public RateLimitingRequestThrottler(DriverContext context) {
        this(context, System::nanoTime);
    }

    @VisibleForTesting
    RateLimitingRequestThrottler(DriverContext context, NanoClock clock) {
        this.logPrefix = context.getSessionName();
        this.clock = clock;
        DriverExecutionProfile config = context.getConfig().getDefaultProfile();
        this.maxRequestsPerSecond = config.getInt(DefaultDriverOption.REQUEST_THROTTLER_MAX_REQUESTS_PER_SECOND);
        this.maxQueueSize = config.getInt(DefaultDriverOption.REQUEST_THROTTLER_MAX_QUEUE_SIZE);
        Duration drainInterval = config.getDuration(DefaultDriverOption.REQUEST_THROTTLER_DRAIN_INTERVAL);
        this.drainIntervalNanos = drainInterval.toNanos();
        this.lastUpdateNanos = clock.nanoTime();
        this.storedPermits = this.maxRequestsPerSecond;
        this.scheduler = ((InternalDriverContext)context).getNettyOptions().adminEventExecutorGroup().next();
        LOG.debug("[{}] Initializing with maxRequestsPerSecond = {}, maxQueueSize = {}, drainInterval = {}", new Object[]{this.logPrefix, this.maxRequestsPerSecond, this.maxQueueSize, drainInterval});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void register(@NonNull Throttled request) {
        long now = this.clock.nanoTime();
        this.lock.lock();
        try {
            if (this.closed) {
                LOG.trace("[{}] Rejecting request after shutdown", (Object)this.logPrefix);
                RateLimitingRequestThrottler.fail(request, "The session is shutting down");
            } else if (this.queue.isEmpty() && this.acquire(now, 1) == 1) {
                LOG.trace("[{}] Starting newly registered request", (Object)this.logPrefix);
                request.onThrottleReady(false);
            } else if (this.queue.size() < this.maxQueueSize) {
                LOG.trace("[{}] Enqueuing request", (Object)this.logPrefix);
                if (this.queue.isEmpty()) {
                    this.scheduler.schedule(this::drain, this.drainIntervalNanos, TimeUnit.NANOSECONDS);
                }
                this.queue.add(request);
            } else {
                LOG.trace("[{}] Rejecting request because of full queue", (Object)this.logPrefix);
                RateLimitingRequestThrottler.fail(request, String.format("The session has reached its maximum capacity (requests/s: %d, queue size: %d)", this.maxRequestsPerSecond, this.maxQueueSize));
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void drain() {
        assert (this.scheduler.inEventLoop());
        long now = this.clock.nanoTime();
        this.lock.lock();
        try {
            if (this.closed || this.queue.isEmpty()) {
                return;
            }
            int toDequeue = this.acquire(now, this.queue.size());
            LOG.trace("[{}] Dequeuing {}/{} elements", new Object[]{this.logPrefix, toDequeue, this.queue.size()});
            for (int i = 0; i < toDequeue; ++i) {
                LOG.trace("[{}] Starting dequeued request", (Object)this.logPrefix);
                this.queue.poll().onThrottleReady(true);
            }
            if (!this.queue.isEmpty()) {
                LOG.trace("[{}] {} elements remaining in queue, rescheduling drain task", (Object)this.logPrefix, (Object)this.queue.size());
                this.scheduler.schedule(this::drain, this.drainIntervalNanos, TimeUnit.NANOSECONDS);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void signalSuccess(@NonNull Throttled request) {
    }

    @Override
    public void signalError(@NonNull Throttled request, @NonNull Throwable error) {
    }

    @Override
    public void signalTimeout(@NonNull Throttled request) {
        this.lock.lock();
        try {
            if (!this.closed && this.queue.remove(request)) {
                LOG.trace("[{}] Removing timed out request from the queue", (Object)this.logPrefix);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void signalCancel(@NonNull Throttled request) {
        this.lock.lock();
        try {
            if (!this.closed && this.queue.remove(request)) {
                LOG.trace("[{}] Removing cancelled request from the queue", (Object)this.logPrefix);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void close() {
        this.lock.lock();
        try {
            this.closed = true;
            LOG.debug("[{}] Rejecting {} queued requests after shutdown", (Object)this.logPrefix, (Object)this.queue.size());
            for (Throttled request : this.queue) {
                RateLimitingRequestThrottler.fail(request, "The session is shutting down");
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private int acquire(long currentTimeNanos, int wantedPermits) {
        assert (this.lock.isHeldByCurrentThread() && !this.closed);
        long elapsedNanos = currentTimeNanos - this.lastUpdateNanos;
        if (elapsedNanos >= 1000000000L) {
            this.storedPermits = this.maxRequestsPerSecond;
            this.lastUpdateNanos = currentTimeNanos;
        } else if (elapsedNanos > 0L) {
            int createdPermits = (int)(elapsedNanos * (long)this.maxRequestsPerSecond / 1000000000L);
            if (createdPermits > 0) {
                this.lastUpdateNanos = currentTimeNanos;
            }
            this.storedPermits = Math.min(this.storedPermits + createdPermits, this.maxRequestsPerSecond);
        }
        int returned = this.storedPermits >= wantedPermits ? wantedPermits : this.storedPermits;
        this.storedPermits = Math.max(this.storedPermits - wantedPermits, 0);
        return returned;
    }

    public int getQueueSize() {
        this.lock.lock();
        try {
            int n = this.queue.size();
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    @VisibleForTesting
    int getStoredPermits() {
        this.lock.lock();
        try {
            int n = this.storedPermits;
            return n;
        }
        finally {
            this.lock.unlock();
        }
    }

    @VisibleForTesting
    Deque<Throttled> getQueue() {
        this.lock.lock();
        try {
            Deque<Throttled> deque = this.queue;
            return deque;
        }
        finally {
            this.lock.unlock();
        }
    }

    private static void fail(Throttled request, String message) {
        request.onThrottleFailure(new RequestThrottlingException(message));
    }
}

